// Renuma Real Estate - Installer JavaScript
let currentStep = 1;
const totalSteps = 6;
let installData = {};

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    updateProgress();
    if (currentStep === 2) {
        checkRequirements();
    }
});

// Navigation Functions
function nextStep() {
    if (currentStep === 2 && !validateRequirements()) {
        return;
    }
    
    if (currentStep === 4) {
        // Collect configuration data
        const siteForm = document.getElementById('siteForm');
        const formData = new FormData(siteForm);
        installData.config = Object.fromEntries(formData);
    }
    
    if (currentStep === 4) {
        // Move to installation step
        currentStep++;
        updateStepDisplay();
        updateProgress();
        startInstallation();
        return;
    }
    
    if (currentStep < totalSteps) {
        currentStep++;
        updateStepDisplay();
        updateProgress();
        
        if (currentStep === 2) {
            checkRequirements();
        }
    }
}

function prevStep() {
    if (currentStep > 1) {
        currentStep--;
        updateStepDisplay();
        updateProgress();
    }
}

function updateStepDisplay() {
    // Hide all steps
    document.querySelectorAll('.step').forEach(step => {
        step.classList.remove('active');
    });
    
    // Show current step
    document.getElementById('step' + currentStep).classList.add('active');
    
    // Update step indicators
    document.querySelectorAll('.step-indicator-item').forEach((item, index) => {
        item.classList.remove('active', 'completed');
        if (index + 1 === currentStep) {
            item.classList.add('active');
        } else if (index + 1 < currentStep) {
            item.classList.add('completed');
        }
    });
}

function updateProgress() {
    const progress = (currentStep / totalSteps) * 100;
    document.getElementById('progressFill').style.width = progress + '%';
}

// Requirements Check
async function checkRequirements() {
    const reqDiv = document.getElementById('requirements');
    reqDiv.innerHTML = '<div style="text-align:center;padding:20px;"><div class="loading"></div><p>Checking requirements...</p></div>';
    
    try {
        const response = await fetch('check_requirements.php');
        const requirements = await response.json();
        
        let html = '';
        let allPassed = true;
        
        requirements.forEach(req => {
            const status = req.status === 'pass' ? 'pass' : 
                          req.status === 'warning' ? 'warning' : 'fail';
            
            if (status === 'fail') allPassed = false;
            
            const icon = status === 'pass' ? '✓' : 
                        status === 'warning' ? '⚠' : '✗';
            
            html += `
                <div class="requirement ${status}">
                    <div class="requirement-icon">${icon}</div>
                    <div class="requirement-text">
                        <div class="requirement-title">${req.name}</div>
                        <div class="requirement-desc">${req.message}</div>
                    </div>
                </div>
            `;
        });
        
        reqDiv.innerHTML = html;
        document.getElementById('reqNextBtn').disabled = !allPassed;
        
        if (!allPassed) {
            reqDiv.innerHTML += `
                <div class="alert alert-danger">
                    <strong>❌ Requirements Not Met</strong><br>
                    Please fix the failed requirements above before continuing. Contact your hosting provider if you need help.
                </div>
            `;
        } else {
            reqDiv.innerHTML += `
                <div class="alert alert-success">
                    <strong>✅ All Requirements Met!</strong><br>
                    Your server is ready for installation. Click Next to continue.
                </div>
            `;
        }
        
    } catch (error) {
        reqDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>Error:</strong> Could not check requirements. ${error.message}
            </div>
        `;
    }
}

function validateRequirements() {
    return !document.getElementById('reqNextBtn').disabled;
}

// Database Testing
async function testDatabase() {
    const form = document.getElementById('dbForm');
    const formData = new FormData(form);
    const resultDiv = document.getElementById('dbTestResult');
    
    resultDiv.innerHTML = '<div style="text-align:center;padding:20px;"><div class="loading"></div><p>Testing database connection...</p></div>';
    
    try {
        const response = await fetch('test_database.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            resultDiv.innerHTML = `
                <div class="alert alert-success">
                    <strong>✅ Connection Successful!</strong><br>
                    Database connection established successfully.
                </div>
            `;
            
            // Save database config
            installData.database = Object.fromEntries(formData);
            
            // Move to next step after 1 second
            setTimeout(() => {
                nextStep();
            }, 1000);
            
        } else {
            resultDiv.innerHTML = `
                <div class="alert alert-danger">
                    <strong>❌ Connection Failed</strong><br>
                    ${result.message}
                </div>
            `;
        }
        
    } catch (error) {
        resultDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>Error:</strong> ${error.message}
            </div>
        `;
    }
}

// Installation Process
async function startInstallation() {
    const logDiv = document.getElementById('installLog');
    const errorDiv = document.getElementById('installError');
    
    logDiv.innerHTML = '';
    errorDiv.innerHTML = '';
    
    addLog('🚀 Starting installation process...');
    
    const steps = [
        { name: 'Creating environment file', action: 'create_env' },
        { name: 'Generating application key', action: 'generate_key' },
        { name: 'Setting up database', action: 'setup_database' },
        { name: 'Running migrations', action: 'run_migrations' },
        { name: 'Seeding sample data', action: 'seed_data' },
        { name: 'Creating storage link', action: 'storage_link' },
        { name: 'Optimizing application', action: 'optimize' },
        { name: 'Setting permissions', action: 'set_permissions' },
        { name: 'Finalizing installation', action: 'finalize' }
    ];
    
    try {
        for (let i = 0; i < steps.length; i++) {
            const step = steps[i];
            
            // Skip demo data if not selected
            if (step.action === 'seed_data' && installData.config.install_demo === 'no') {
                addLog(`⊘ Skipping: ${step.name}`);
                continue;
            }
            
            addLog(`⏳ ${step.name}...`);
            
            const response = await fetch('install.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    action: step.action,
                    database: installData.database,
                    config: installData.config
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                addLog(`✅ ${step.name} - Complete`);
                
                if (result.message) {
                    addLog(`   ${result.message}`);
                }
            } else {
                throw new Error(result.message || 'Installation step failed');
            }
            
            // Small delay for better UX
            await sleep(300);
        }
        
        addLog('');
        addLog('🎉 Installation completed successfully!');
        addLog('✅ Your Renuma Real Estate platform is ready to use.');
        
        // Update final step with site URL
        document.getElementById('adminUrl').textContent = installData.config.app_url + '/admin';
        
        // Show sample data counts if installed
        if (installData.config.install_demo === 'yes') {
            document.getElementById('propCount').textContent = '5 sample properties';
            document.getElementById('agentCount').textContent = '4 team members';
            document.getElementById('leadCount').textContent = '3 sample inquiries';
        } else {
            document.getElementById('propCount').textContent = '0 properties';
            document.getElementById('agentCount').textContent = '0 agents';
            document.getElementById('leadCount').textContent = '0 leads';
        }
        
        document.getElementById('installComplete').style.display = 'flex';
        
    } catch (error) {
        addLog(`❌ Error: ${error.message}`);
        errorDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>❌ Installation Failed</strong><br>
                ${error.message}<br><br>
                Please check the installation log above for details and try again.
            </div>
        `;
    }
}

function addLog(message) {
    const logDiv = document.getElementById('installLog');
    const line = document.createElement('div');
    line.textContent = message;
    logDiv.appendChild(line);
    logDiv.scrollTop = logDiv.scrollHeight;
}

function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

// Utility Functions
function showAlert(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type}`;
    alertDiv.innerHTML = message;
    
    const content = document.querySelector('.content');
    content.insertBefore(alertDiv, content.firstChild);
    
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}
