<?php
/**
 * Renuma Real Estate - System Requirements Checker
 */

header('Content-Type: application/json');

$requirements = [];

// PHP Version Check
$phpVersion = phpversion();
$phpRequired = '8.1.0';
$requirements[] = [
    'name' => 'PHP Version',
    'status' => version_compare($phpVersion, $phpRequired, '>=') ? 'pass' : 'fail',
    'message' => "Current: $phpVersion | Required: $phpRequired or higher"
];

// PHP Extensions
$extensions = [
    'openssl' => 'OpenSSL PHP Extension',
    'pdo' => 'PDO PHP Extension',
    'pdo_mysql' => 'PDO MySQL Extension',
    'mbstring' => 'Mbstring PHP Extension',
    'tokenizer' => 'Tokenizer PHP Extension',
    'xml' => 'XML PHP Extension',
    'ctype' => 'Ctype PHP Extension',
    'json' => 'JSON PHP Extension',
    'bcmath' => 'BCMath PHP Extension',
    'fileinfo' => 'Fileinfo PHP Extension',
    'gd' => 'GD PHP Extension',
    'curl' => 'cURL PHP Extension'
];

foreach ($extensions as $ext => $name) {
    $requirements[] = [
        'name' => $name,
        'status' => extension_loaded($ext) ? 'pass' : 'fail',
        'message' => extension_loaded($ext) ? 'Installed' : 'Not installed - Please enable this extension'
    ];
}

// Directory Permissions
$directories = [
    '../backend/storage' => 'Storage Directory',
    '../backend/bootstrap/cache' => 'Bootstrap Cache Directory'
];

foreach ($directories as $dir => $name) {
    $writable = is_writable($dir);
    $requirements[] = [
        'name' => $name . ' Writable',
        'status' => $writable ? 'pass' : 'fail',
        'message' => $writable ? 'Writable (permissions OK)' : 'Not writable - Set permissions to 775'
    ];
}

// .env file check
$envExists = file_exists('../backend/.env');
$requirements[] = [
    'name' => 'Environment File',
    'status' => !$envExists ? 'pass' : 'warning',
    'message' => !$envExists ? 'Ready to create' : '.env file already exists - will be overwritten'
];

// Composer check
$vendorExists = file_exists('../backend/vendor/autoload.php');
$requirements[] = [
    'name' => 'Composer Dependencies',
    'status' => $vendorExists ? 'pass' : 'fail',
    'message' => $vendorExists ? 'Dependencies installed' : 'Run: composer install --no-dev'
];

// MySQL Functions
$requirements[] = [
    'name' => 'MySQL Support',
    'status' => (function_exists('mysqli_connect') || extension_loaded('pdo_mysql')) ? 'pass' : 'fail',
    'message' => (function_exists('mysqli_connect') || extension_loaded('pdo_mysql')) ? 'Available' : 'MySQL support not found'
];

// mod_rewrite (if Apache)
if (function_exists('apache_get_modules')) {
    $modRewrite = in_array('mod_rewrite', apache_get_modules());
    $requirements[] = [
        'name' => 'Apache mod_rewrite',
        'status' => $modRewrite ? 'pass' : 'warning',
        'message' => $modRewrite ? 'Enabled' : 'May need to be enabled for clean URLs'
    ];
}

// PHP Memory Limit
$memoryLimit = ini_get('memory_limit');
$requirements[] = [
    'name' => 'PHP Memory Limit',
    'status' => (intval($memoryLimit) >= 128 || $memoryLimit === '-1') ? 'pass' : 'warning',
    'message' => "Current: $memoryLimit | Recommended: 128M or higher"
];

// Upload Max Filesize
$uploadMax = ini_get('upload_max_filesize');
$requirements[] = [
    'name' => 'Upload Max Filesize',
    'status' => (intval($uploadMax) >= 20) ? 'pass' : 'warning',
    'message' => "Current: $uploadMax | Recommended: 20M or higher for property images"
];

// Max Execution Time
$maxExecTime = ini_get('max_execution_time');
$requirements[] = [
    'name' => 'Max Execution Time',
    'status' => ($maxExecTime >= 300 || $maxExecTime == 0) ? 'pass' : 'warning',
    'message' => "Current: {$maxExecTime}s | Recommended: 300s or higher"
];

echo json_encode($requirements);
