import React, { useState, useEffect } from 'react';
import {
  Home, Users, FileText, MessageSquare, Settings, 
  TrendingUp, DollarSign, Eye, Plus, Download,
  BarChart3, PieChart, Activity
} from 'lucide-react';
import {
  LineChart, Line, BarChart, Bar, PieChart as RePieChart, Pie, Cell,
  XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer
} from 'recharts';

export default function AdminDashboard() {
  const [stats, setStats] = useState(null);
  const [loading, setLoading] = useState(true);
  const [timeRange, setTimeRange] = useState('30days');

  useEffect(() => {
    fetchDashboardData();
  }, [timeRange]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/admin/dashboard', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('admin_token')}`
        }
      });
      const data = await response.json();
      if (data.success) {
        setStats(data.data);
      }
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-gold-500"></div>
      </div>
    );
  }

  const COLORS = ['#D4AF37', '#1A1A1A', '#4A90E2', '#10B981', '#EF4444'];

  return (
    <div className="admin-dashboard p-6 bg-gray-50 min-h-screen">
      {/* Header */}
      <div className="mb-8">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Dashboard</h1>
            <p className="text-gray-600 mt-1">Welcome back! Here's what's happening today.</p>
          </div>
          <div className="flex gap-3">
            <select
              value={timeRange}
              onChange={(e) => setTimeRange(e.target.value)}
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-gold-400 focus:border-transparent"
            >
              <option value="7days">Last 7 Days</option>
              <option value="30days">Last 30 Days</option>
              <option value="90days">Last 90 Days</option>
              <option value="year">This Year</option>
            </select>
            <button className="px-4 py-2 bg-gold-500 text-white rounded-lg hover:bg-gold-600 transition-colors flex items-center gap-2">
              <Download className="w-4 h-4" />
              Export Report
            </button>
          </div>
        </div>
      </div>

      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        {/* Total Properties */}
        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-gold-500">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-gold-100 rounded-lg flex items-center justify-center">
              <Home className="w-6 h-6 text-gold-600" />
            </div>
            <span className="text-sm text-green-600 font-medium">+12%</span>
          </div>
          <h3 className="text-2xl font-bold text-gray-900">
            {stats?.overview?.total_properties || 0}
          </h3>
          <p className="text-gray-600 text-sm mt-1">Total Properties</p>
        </div>

        {/* Active Listings */}
        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-blue-500">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
              <Activity className="w-6 h-6 text-blue-600" />
            </div>
            <span className="text-sm text-green-600 font-medium">+8%</span>
          </div>
          <h3 className="text-2xl font-bold text-gray-900">
            {stats?.overview?.active_properties || 0}
          </h3>
          <p className="text-gray-600 text-sm mt-1">Active Listings</p>
        </div>

        {/* Total Leads */}
        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-purple-500">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
              <MessageSquare className="w-6 h-6 text-purple-600" />
            </div>
            <span className="text-sm text-green-600 font-medium">+23%</span>
          </div>
          <h3 className="text-2xl font-bold text-gray-900">
            {stats?.overview?.total_leads || 0}
          </h3>
          <p className="text-gray-600 text-sm mt-1">Total Leads</p>
        </div>

        {/* Revenue */}
        <div className="bg-white rounded-xl shadow-sm p-6 border-l-4 border-green-500">
          <div className="flex items-center justify-between mb-4">
            <div className="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
              <DollarSign className="w-6 h-6 text-green-600" />
            </div>
            <span className="text-sm text-green-600 font-medium">+15%</span>
          </div>
          <h3 className="text-2xl font-bold text-gray-900">
            ৳{((stats?.revenue?.total_sales_value || 0) / 10000000).toFixed(2)}Cr
          </h3>
          <p className="text-gray-600 text-sm mt-1">Total Sales Value</p>
        </div>
      </div>

      {/* Charts Row */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
        {/* Monthly Statistics */}
        <div className="bg-white rounded-xl shadow-sm p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Monthly Performance</h3>
            <BarChart3 className="w-5 h-5 text-gray-400" />
          </div>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={stats?.monthly_stats || []}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="month" />
              <YAxis />
              <Tooltip />
              <Legend />
              <Line type="monotone" dataKey="properties_added" stroke="#D4AF37" strokeWidth={2} />
              <Line type="monotone" dataKey="leads_received" stroke="#4A90E2" strokeWidth={2} />
              <Line type="monotone" dataKey="properties_sold" stroke="#10B981" strokeWidth={2} />
            </LineChart>
          </ResponsiveContainer>
        </div>

        {/* Property Distribution */}
        <div className="bg-white rounded-xl shadow-sm p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Property Distribution</h3>
            <PieChart className="w-5 h-5 text-gray-400" />
          </div>
          <ResponsiveContainer width="100%" height={300}>
            <RePieChart>
              <Pie
                data={stats?.properties?.by_type || []}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                outerRadius={100}
                fill="#8884d8"
                dataKey="count"
              >
                {stats?.properties?.by_type?.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                ))}
              </Pie>
              <Tooltip />
            </RePieChart>
          </ResponsiveContainer>
        </div>
      </div>

      {/* Tables Row */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Recent Properties */}
        <div className="bg-white rounded-xl shadow-sm p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Recent Properties</h3>
            <a href="/admin/properties" className="text-gold-500 hover:text-gold-600 text-sm font-medium">
              View All
            </a>
          </div>
          <div className="space-y-4">
            {stats?.recent_activities?.recent_properties?.map((property) => (
              <div key={property.id} className="flex items-center gap-4 p-3 hover:bg-gray-50 rounded-lg transition-colors">
                <img
                  src={property.featured_image || '/images/placeholder.jpg'}
                  alt={property.title}
                  className="w-16 h-16 rounded-lg object-cover"
                />
                <div className="flex-1 min-w-0">
                  <h4 className="font-medium text-gray-900 truncate">{property.title}</h4>
                  <p className="text-sm text-gray-500">{property.location?.name}</p>
                </div>
                <div className="text-right">
                  <p className="font-semibold text-gray-900">৳{(property.price / 100000).toFixed(2)}L</p>
                  <span className={`text-xs px-2 py-1 rounded-full ${
                    property.status === 'available' ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-700'
                  }`}>
                    {property.status}
                  </span>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Recent Leads */}
        <div className="bg-white rounded-xl shadow-sm p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-semibold text-gray-900">Recent Leads</h3>
            <a href="/admin/leads" className="text-gold-500 hover:text-gold-600 text-sm font-medium">
              View All
            </a>
          </div>
          <div className="space-y-4">
            {stats?.recent_activities?.recent_leads?.map((lead) => (
              <div key={lead.id} className="flex items-center gap-4 p-3 hover:bg-gray-50 rounded-lg transition-colors">
                <div className="w-10 h-10 bg-gold-100 rounded-full flex items-center justify-center">
                  <Users className="w-5 h-5 text-gold-600" />
                </div>
                <div className="flex-1 min-w-0">
                  <h4 className="font-medium text-gray-900">{lead.name}</h4>
                  <p className="text-sm text-gray-500 truncate">{lead.email}</p>
                </div>
                <div className="text-right">
                  <span className={`text-xs px-2 py-1 rounded-full ${
                    lead.status === 'new' ? 'bg-blue-100 text-blue-700' :
                    lead.status === 'contacted' ? 'bg-yellow-100 text-yellow-700' :
                    lead.status === 'converted' ? 'bg-green-100 text-green-700' :
                    'bg-gray-100 text-gray-700'
                  }`}>
                    {lead.status}
                  </span>
                  <p className="text-xs text-gray-400 mt-1">
                    {new Date(lead.created_at).toLocaleDateString()}
                  </p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Quick Actions */}
      <div className="mt-8 bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Quick Actions</h3>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          <a
            href="/admin/properties/create"
            className="flex items-center gap-3 p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-gold-500 hover:bg-gold-50 transition-all group"
          >
            <Plus className="w-5 h-5 text-gray-400 group-hover:text-gold-500" />
            <span className="font-medium text-gray-700 group-hover:text-gold-500">Add Property</span>
          </a>
          <a
            href="/admin/agents/create"
            className="flex items-center gap-3 p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-blue-500 hover:bg-blue-50 transition-all group"
          >
            <Plus className="w-5 h-5 text-gray-400 group-hover:text-blue-500" />
            <span className="font-medium text-gray-700 group-hover:text-blue-500">Add Agent</span>
          </a>
          <a
            href="/admin/blog/create"
            className="flex items-center gap-3 p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-purple-500 hover:bg-purple-50 transition-all group"
          >
            <Plus className="w-5 h-5 text-gray-400 group-hover:text-purple-500" />
            <span className="font-medium text-gray-700 group-hover:text-purple-500">New Post</span>
          </a>
          <a
            href="/admin/settings"
            className="flex items-center gap-3 p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-gray-500 hover:bg-gray-50 transition-all group"
          >
            <Settings className="w-5 h-5 text-gray-400 group-hover:text-gray-500" />
            <span className="font-medium text-gray-700 group-hover:text-gray-500">Settings</span>
          </a>
        </div>
      </div>
    </div>
  );
}
