<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Property;
use App\Models\PropertyType;
use App\Models\Location;
use App\Models\Agent;
use App\Models\Feature;
use App\Models\Lead;
use App\Models\BlogPost;
use App\Models\BlogCategory;
use App\Models\Testimonial;
use App\Models\User;
use Illuminate\Support\Str;
use Carbon\Carbon;

class DummyDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run()
    {
        echo "🌱 Starting Renuma Real Estate Data Seeding...\n\n";

        $this->seedUsers();
        $this->seedLocations();
        $this->seedPropertyTypes();
        $this->seedFeatures();
        $this->seedAgents();
        $this->seedProperties();
        $this->seedLeads();
        $this->seedBlog();
        $this->seedTestimonials();

        echo "\n✅ Seeding completed successfully!\n";
    }

    private function seedUsers()
    {
        echo "👤 Creating users...\n";

        // Admin User
        User::create([
            'name' => 'Admin User',
            'email' => 'admin@renumarealestate.com',
            'password' => bcrypt('Admin@123'),
            'role' => 'admin',
            'phone' => '+8801315333100',
            'status' => 'active',
        ]);

        // Agent Users
        User::create([
            'name' => 'Savannah Nguyen',
            'email' => 'savannah@renumarealestate.com',
            'password' => bcrypt('Agent@123'),
            'role' => 'agent',
            'phone' => '+8801712345671',
            'status' => 'active',
        ]);

        User::create([
            'name' => 'Annette Black',
            'email' => 'annette@renumarealestate.com',
            'password' => bcrypt('Agent@123'),
            'role' => 'agent',
            'phone' => '+8801712345672',
            'status' => 'active',
        ]);

        echo "   ✓ Created 3 users\n";
    }

    private function seedLocations()
    {
        echo "📍 Creating locations...\n";

        $locations = [
            ['name' => 'Dhaka', 'slug' => 'dhaka', 'type' => 'city', 'latitude' => 23.8103, 'longitude' => 90.4125],
            ['name' => 'Gulshan', 'slug' => 'gulshan', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.7805, 'longitude' => 90.4147, 'is_featured' => true],
            ['name' => 'Banani', 'slug' => 'banani', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.7937, 'longitude' => 90.4066, 'is_featured' => true],
            ['name' => 'Dhanmondi', 'slug' => 'dhanmondi', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.7461, 'longitude' => 90.3742, 'is_featured' => true],
            ['name' => 'Mirpur', 'slug' => 'mirpur', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.8223, 'longitude' => 90.3654],
            ['name' => 'Uttara', 'slug' => 'uttara', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.8759, 'longitude' => 90.3795, 'is_featured' => true],
            ['name' => 'Bashundhara', 'slug' => 'bashundhara', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.8225, 'longitude' => 90.4274, 'is_featured' => true],
            ['name' => 'Mohakhali', 'slug' => 'mohakhali', 'type' => 'area', 'parent_id' => 1, 'latitude' => 23.7808, 'longitude' => 90.4080],
        ];

        foreach ($locations as $location) {
            Location::create($location);
        }

        echo "   ✓ Created " . count($locations) . " locations\n";
    }

    private function seedPropertyTypes()
    {
        echo "🏠 Creating property types...\n";

        $types = [
            ['name' => 'Apartment', 'slug' => 'apartment', 'icon' => 'fa-building'],
            ['name' => 'Villa', 'slug' => 'villa', 'icon' => 'fa-home'],
            ['name' => 'Penthouse', 'slug' => 'penthouse', 'icon' => 'fa-city'],
            ['name' => 'Land', 'slug' => 'land', 'icon' => 'fa-map'],
            ['name' => 'Commercial', 'slug' => 'commercial', 'icon' => 'fa-store'],
            ['name' => 'Office Space', 'slug' => 'office-space', 'icon' => 'fa-briefcase'],
        ];

        foreach ($types as $type) {
            PropertyType::create($type);
        }

        echo "   ✓ Created " . count($types) . " property types\n";
    }

    private function seedFeatures()
    {
        echo "✨ Creating features...\n";

        $features = [
            // Comfort
            ['name' => 'Air Conditioning', 'icon' => 'fa-snowflake', 'category' => 'Comfort'],
            ['name' => 'Heating', 'icon' => 'fa-fire', 'category' => 'Comfort'],
            ['name' => 'Fireplace', 'icon' => 'fa-fire-alt', 'category' => 'Comfort'],
            
            // Outdoor
            ['name' => 'Swimming Pool', 'icon' => 'fa-swimming-pool', 'category' => 'Outdoor'],
            ['name' => 'Garden', 'icon' => 'fa-leaf', 'category' => 'Outdoor'],
            ['name' => 'Balcony', 'icon' => 'fa-warehouse', 'category' => 'Outdoor'],
            ['name' => 'Terrace', 'icon' => 'fa-home', 'category' => 'Outdoor'],
            
            // Amenities
            ['name' => 'Gym', 'icon' => 'fa-dumbbell', 'category' => 'Amenities'],
            ['name' => 'Sauna', 'icon' => 'fa-hot-tub', 'category' => 'Amenities'],
            ['name' => 'Jacuzzi', 'icon' => 'fa-bath', 'category' => 'Amenities'],
            
            // Security
            ['name' => 'Security System', 'icon' => 'fa-shield-alt', 'category' => 'Security'],
            ['name' => 'CCTV', 'icon' => 'fa-video', 'category' => 'Security'],
            ['name' => 'Intercom', 'icon' => 'fa-phone', 'category' => 'Security'],
            
            // Building
            ['name' => 'Elevator', 'icon' => 'fa-elevator', 'category' => 'Building'],
            ['name' => 'Parking', 'icon' => 'fa-car', 'category' => 'Building'],
            ['name' => 'Storage', 'icon' => 'fa-boxes', 'category' => 'Building'],
            
            // Technology
            ['name' => 'WiFi', 'icon' => 'fa-wifi', 'category' => 'Technology'],
            ['name' => 'Smart Home', 'icon' => 'fa-mobile', 'category' => 'Technology'],
            ['name' => 'Solar Panels', 'icon' => 'fa-solar-panel', 'category' => 'Technology'],
            
            // Other
            ['name' => 'Furnished', 'icon' => 'fa-couch', 'category' => 'Features'],
            ['name' => 'Pet Friendly', 'icon' => 'fa-paw', 'category' => 'Features'],
            ['name' => 'Laundry Room', 'icon' => 'fa-tshirt', 'category' => 'Features'],
        ];

        foreach ($features as $feature) {
            Feature::create($feature);
        }

        echo "   ✓ Created " . count($features) . " features\n";
    }

    private function seedAgents()
    {
        echo "👔 Creating agents...\n";

        $agents = [
            [
                'user_id' => 2,
                'name' => 'Savannah Nguyen',
                'email' => 'savannah@renumarealestate.com',
                'phone' => '+8801712345671',
                'whatsapp' => '+8801712345671',
                'designation' => 'Senior Property Consultant',
                'bio' => 'With over 10 years of experience in luxury real estate, Savannah has helped hundreds of clients find their dream homes in Dhaka.',
                'experience_years' => 10,
                'properties_sold' => 150,
                'is_featured' => true,
                'status' => 'active',
            ],
            [
                'user_id' => 3,
                'name' => 'Annette Black',
                'email' => 'annette@renumarealestate.com',
                'phone' => '+8801712345672',
                'whatsapp' => '+8801712345672',
                'designation' => 'Property Specialist',
                'bio' => 'Specializing in commercial properties and investment opportunities. Annette brings expertise and market knowledge to every transaction.',
                'experience_years' => 7,
                'properties_sold' => 95,
                'is_featured' => true,
                'status' => 'active',
            ],
            [
                'name' => 'Kathryn Murphy',
                'email' => 'kathryn@renumarealestate.com',
                'phone' => '+8801712345673',
                'whatsapp' => '+8801712345673',
                'designation' => 'Luxury Property Advisor',
                'bio' => 'Expert in high-end residential properties with a keen eye for luxury details and exceptional customer service.',
                'experience_years' => 8,
                'properties_sold' => 120,
                'is_featured' => true,
                'status' => 'active',
            ],
            [
                'name' => 'David Hardson',
                'email' => 'david@renumarealestate.com',
                'phone' => '+8801712345674',
                'whatsapp' => '+8801712345674',
                'designation' => 'Real Estate Consultant',
                'bio' => 'Passionate about helping first-time homebuyers navigate the real estate market with confidence.',
                'experience_years' => 5,
                'properties_sold' => 70,
                'is_featured' => false,
                'status' => 'active',
            ],
        ];

        foreach ($agents as $agent) {
            Agent::create($agent);
        }

        echo "   ✓ Created " . count($agents) . " agents\n";
    }

    private function seedProperties()
    {
        echo "🏘️  Creating properties...\n";

        $properties = [
            [
                'title' => 'Luxury Penthouse in Gulshan 2',
                'property_type_id' => 3, // Penthouse
                'location_id' => 2, // Gulshan
                'agent_id' => 1,
                'description' => 'Stunning luxury penthouse with panoramic city views. Features include marble flooring, imported fittings, and state-of-the-art amenities. Located in the heart of Gulshan, this property offers unparalleled luxury living.',
                'short_description' => 'Luxury penthouse with city views and premium amenities',
                'address' => 'Road 112, Gulshan 2, Dhaka',
                'latitude' => 23.7918,
                'longitude' => 90.4152,
                'price' => 55000000, // 5.5 Crore
                'price_type' => 'negotiable',
                'status' => 'available',
                'purpose' => 'sale',
                'bedrooms' => 4,
                'bathrooms' => 5,
                'area_sqft' => 3500,
                'area_type' => 'Built-up',
                'floors' => 1,
                'year_built' => 2022,
                'parking_spaces' => 3,
                'is_featured' => true,
                'is_verified' => true,
                'published_at' => now(),
            ],
            [
                'title' => 'Modern Apartment in Banani DOHS',
                'property_type_id' => 1, // Apartment
                'location_id' => 3, // Banani
                'agent_id' => 2,
                'description' => 'Beautifully designed 3-bedroom apartment in prime Banani location. Features modern kitchen, spacious living areas, and excellent natural lighting. Perfect for families.',
                'short_description' => 'Modern 3-bedroom apartment with excellent amenities',
                'address' => 'DOHS Banani, Road 11, Dhaka',
                'latitude' => 23.7945,
                'longitude' => 90.4070,
                'price' => 28000000, // 2.8 Crore
                'status' => 'available',
                'purpose' => 'sale',
                'bedrooms' => 3,
                'bathrooms' => 3,
                'area_sqft' => 1860,
                'area_type' => 'Built-up',
                'floors' => 1,
                'year_built' => 2021,
                'parking_spaces' => 2,
                'is_featured' => true,
                'is_verified' => true,
                'published_at' => now(),
            ],
            [
                'title' => 'Spacious Villa in Bashundhara R/A',
                'property_type_id' => 2, // Villa
                'location_id' => 7, // Bashundhara
                'agent_id' => 3,
                'description' => 'Magnificent villa with private garden and pool. Features 5 spacious bedrooms, modern architecture, and premium finishes throughout. Ideal for luxury living.',
                'short_description' => 'Luxury villa with private garden and swimming pool',
                'address' => 'Block G, Road 7, Bashundhara R/A, Dhaka',
                'latitude' => 23.8236,
                'longitude' => 90.4285,
                'price' => 85000000, // 8.5 Crore
                'price_type' => 'fixed',
                'status' => 'available',
                'purpose' => 'sale',
                'bedrooms' => 5,
                'bathrooms' => 6,
                'area_sqft' => 4500,
                'area_type' => 'Total',
                'floors' => 2,
                'year_built' => 2023,
                'parking_spaces' => 4,
                'is_featured' => true,
                'is_verified' => true,
                'is_urgent' => true,
                'published_at' => now(),
            ],
            [
                'title' => 'Cozy Apartment for Rent in Dhanmondi',
                'property_type_id' => 1, // Apartment
                'location_id' => 4, // Dhanmondi
                'agent_id' => 1,
                'description' => 'Well-maintained 2-bedroom apartment available for rent. Fully furnished with modern amenities, convenient location near schools and shopping centers.',
                'short_description' => 'Fully furnished 2-bedroom apartment in prime location',
                'address' => 'Road 9A, Dhanmondi, Dhaka',
                'latitude' => 23.7465,
                'longitude' => 90.3748,
                'price' => 45000, // Monthly rent
                'status' => 'available',
                'purpose' => 'rent',
                'bedrooms' => 2,
                'bathrooms' => 2,
                'area_sqft' => 1200,
                'floors' => 1,
                'year_built' => 2020,
                'parking_spaces' => 1,
                'is_verified' => true,
                'published_at' => now(),
            ],
            [
                'title' => 'Commercial Space in Uttara',
                'property_type_id' => 5, // Commercial
                'location_id' => 6, // Uttara
                'agent_id' => 2,
                'description' => 'Prime commercial space suitable for office or retail. Located in busy commercial area with high foot traffic. Excellent visibility and parking facilities.',
                'short_description' => 'Prime commercial space with excellent visibility',
                'address' => 'Sector 7, Uttara, Dhaka',
                'latitude' => 23.8765,
                'longitude' => 90.3800,
                'price' => 35000000,
                'status' => 'available',
                'purpose' => 'both',
                'area_sqft' => 2500,
                'floors' => 1,
                'year_built' => 2021,
                'parking_spaces' => 5,
                'is_verified' => true,
                'published_at' => now(),
            ],
        ];

        foreach ($properties as $propertyData) {
            $property = Property::create($propertyData);
            
            // Attach random features
            $featureIds = Feature::inRandomOrder()->limit(rand(5, 10))->pluck('id');
            $property->features()->attach($featureIds);
            
            // Increment property count for location
            Location::find($property->location_id)->increment('property_count');
        }

        echo "   ✓ Created " . count($properties) . " properties\n";
    }

    private function seedLeads()
    {
        echo "📬 Creating leads...\n";

        $leads = [
            [
                'property_id' => 1,
                'agent_id' => 1,
                'name' => 'Ahmed Khan',
                'email' => 'ahmed.khan@example.com',
                'phone' => '+8801712345681',
                'message' => 'I am interested in viewing this penthouse. Please contact me to schedule a visit.',
                'inquiry_type' => 'viewing',
                'status' => 'new',
                'priority' => 'high',
                'created_at' => Carbon::now()->subDays(1),
            ],
            [
                'property_id' => 2,
                'agent_id' => 2,
                'name' => 'Fatima Rahman',
                'email' => 'fatima.r@example.com',
                'phone' => '+8801712345682',
                'message' => 'Is this property still available? What is the final price?',
                'inquiry_type' => 'property',
                'status' => 'contacted',
                'budget_min' => 25000000,
                'budget_max' => 30000000,
                'created_at' => Carbon::now()->subDays(2),
            ],
            [
                'name' => 'Karim Hossain',
                'email' => 'karim.h@example.com',
                'phone' => '+8801712345683',
                'message' => 'Looking for a 3-bedroom apartment in Gulshan or Banani area. Budget around 3 crore.',
                'inquiry_type' => 'general',
                'status' => 'qualified',
                'budget_min' => 28000000,
                'budget_max' => 35000000,
                'preferred_location' => 'Gulshan, Banani',
                'property_type' => 'Apartment',
                'created_at' => Carbon::now()->subDays(3),
            ],
        ];

        foreach ($leads as $lead) {
            Lead::create($lead);
        }

        echo "   ✓ Created " . count($leads) . " leads\n";
    }

    private function seedBlog()
    {
        echo "📝 Creating blog content...\n";

        // Categories
        $categories = [
            ['name' => 'Market Trends', 'slug' => 'market-trends'],
            ['name' => 'Home Buying Tips', 'slug' => 'home-buying-tips'],
            ['name' => 'Investment Guide', 'slug' => 'investment-guide'],
            ['name' => 'Property News', 'slug' => 'property-news'],
        ];

        foreach ($categories as $category) {
            BlogCategory::create($category);
        }

        // Blog Posts
        $posts = [
            [
                'title' => 'Top 10 Neighborhoods in Dhaka for Luxury Living',
                'category_id' => 4,
                'author_id' => 1,
                'excerpt' => 'Discover the most prestigious neighborhoods in Dhaka that offer luxury living and excellent amenities.',
                'content' => 'Dhaka has emerged as a hub for luxury real estate with several neighborhoods offering world-class amenities...',
                'status' => 'published',
                'is_featured' => true,
                'published_at' => Carbon::now()->subDays(5),
            ],
            [
                'title' => 'Essential Checklist for First-Time Home Buyers',
                'category_id' => 2,
                'author_id' => 1,
                'excerpt' => 'A comprehensive guide to help first-time buyers navigate the home buying process with confidence.',
                'content' => 'Buying your first home is an exciting milestone. Here are the essential steps you need to follow...',
                'status' => 'published',
                'is_featured' => true,
                'published_at' => Carbon::now()->subDays(10),
            ],
        ];

        foreach ($posts as $post) {
            BlogPost::create($post);
        }

        echo "   ✓ Created blog content\n";
    }

    private function seedTestimonials()
    {
        echo "💬 Creating testimonials...\n";

        $testimonials = [
            [
                'name' => 'Christine Eve',
                'designation' => 'Founder & CEO',
                'company' => 'TechStart Inc.',
                'rating' => 5,
                'message' => 'Renuma Real Estate helped us find the perfect office space for our expanding team. Their professionalism and market knowledge are exceptional.',
                'is_featured' => true,
                'status' => 'active',
                'sort_order' => 1,
            ],
            [
                'name' => 'Kevin Smith',
                'designation' => 'Business Owner',
                'rating' => 5,
                'message' => 'From the first viewing to the final paperwork, the team at Renuma made our home buying experience smooth and stress-free. Highly recommended!',
                'is_featured' => true,
                'status' => 'active',
                'sort_order' => 2,
            ],
            [
                'name' => 'Jessica Brown',
                'designation' => 'Interior Designer',
                'rating' => 5,
                'message' => 'I found my dream apartment through Renuma Real Estate. The agents were patient, knowledgeable, and truly understood what I was looking for.',
                'is_featured' => true,
                'status' => 'active',
                'sort_order' => 3,
            ],
        ];

        foreach ($testimonials as $testimonial) {
            Testimonial::create($testimonial);
        }

        echo "   ✓ Created " . count($testimonials) . " testimonials\n";
    }
}
