# Renuma Real Estate - Deployment Guide

## 🚀 Quick Start Deployment

### Prerequisites
- Ubuntu 20.04+ / CentOS 7+
- PHP 8.1+
- MySQL 8.0+
- Node.js 18+
- Composer
- Nginx / Apache
- SSL Certificate

---

## 📦 Backend Deployment (Laravel API)

### 1. Server Preparation

```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install PHP 8.1 and extensions
sudo apt install -y php8.1 php8.1-fpm php8.1-mysql php8.1-mbstring \
php8.1-xml php8.1-bcmath php8.1-curl php8.1-gd php8.1-zip \
php8.1-intl php8.1-redis

# Install MySQL
sudo apt install mysql-server -y
sudo mysql_secure_installation

# Install Composer
curl -sS https://getcomposer.org/installer | php
sudo mv composer.phar /usr/local/bin/composer

# Install Node.js & npm
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt install -y nodejs
```

### 2. Clone & Setup Backend

```bash
# Navigate to web directory
cd /var/www

# Clone repository
git clone https://github.com/your-repo/renuma-real-estate.git
cd renuma-real-estate/backend

# Install dependencies
composer install --optimize-autoloader --no-dev

# Set permissions
sudo chown -R www-data:www-data /var/www/renuma-real-estate
sudo chmod -R 755 /var/www/renuma-real-estate
sudo chmod -R 775 storage bootstrap/cache

# Copy environment file
cp .env.example .env

# Generate application key
php artisan key:generate
```

### 3. Database Setup

```bash
# Login to MySQL
mysql -u root -p

# Create database and user
CREATE DATABASE renuma_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'renuma_user'@'localhost' IDENTIFIED BY 'strong_password_here';
GRANT ALL PRIVILEGES ON renuma_db.* TO 'renuma_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;

# Run migrations
php artisan migrate --force

# Seed database
php artisan db:seed --force

# Create storage link
php artisan storage:link
```

### 4. Configure Environment (.env)

```env
APP_NAME="Renuma Real Estate"
APP_ENV=production
APP_KEY=base64:YOUR_APP_KEY
APP_DEBUG=false
APP_URL=https://renumarealestate.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=renuma_db
DB_USERNAME=renuma_user
DB_PASSWORD=your_database_password

# Email Configuration
MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=renumarealestate@gmail.com
MAIL_PASSWORD=your_email_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=renumarealestate@gmail.com
MAIL_FROM_NAME="Renuma Real Estate"

# Google Maps
GOOGLE_MAPS_API_KEY=your_google_maps_api_key

# WhatsApp
WHATSAPP_NUMBER=+8801315333100

# Cache & Queue
CACHE_DRIVER=redis
QUEUE_CONNECTION=redis
SESSION_DRIVER=redis

REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

### 5. Optimize Application

```bash
# Cache configuration
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Optimize autoloader
composer dump-autoload --optimize
```

### 6. Setup Queue Worker (Optional)

```bash
# Create supervisor config
sudo nano /etc/supervisor/conf.d/renuma-worker.conf

# Add this content:
[program:renuma-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /var/www/renuma-real-estate/backend/artisan queue:work --sleep=3 --tries=3
autostart=true
autorestart=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/www/renuma-real-estate/backend/storage/logs/worker.log

# Reload supervisor
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start renuma-worker:*
```

### 7. Setup Scheduler (Cron)

```bash
# Edit crontab
crontab -e

# Add Laravel scheduler
* * * * * cd /var/www/renuma-real-estate/backend && php artisan schedule:run >> /dev/null 2>&1
```

---

## 🌐 Frontend Deployment (Next.js)

### 1. Build Frontend

```bash
cd /var/www/renuma-real-estate/frontend

# Install dependencies
npm ci

# Create environment file
nano .env.local

# Add:
NEXT_PUBLIC_API_URL=https://renumarealestate.com/api
NEXT_PUBLIC_SITE_URL=https://renumarealestate.com
NEXT_PUBLIC_GOOGLE_MAPS_KEY=your_google_maps_api_key
NEXT_PUBLIC_WHATSAPP=+8801315333100

# Build application
npm run build

# Install PM2 for process management
sudo npm install -g pm2

# Start application
pm2 start npm --name "renuma-frontend" -- start
pm2 save
pm2 startup
```

---

## 🔧 Nginx Configuration

### API Configuration

```nginx
# /etc/nginx/sites-available/renuma-api
server {
    listen 80;
    server_name api.renumarealestate.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name api.renumarealestate.com;
    root /var/www/renuma-real-estate/backend/public;

    # SSL Configuration
    ssl_certificate /etc/letsencrypt/live/api.renumarealestate.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/api.renumarealestate.com/privkey.pem;

    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;

    index index.php;
    charset utf-8;

    # Logging
    access_log /var/log/nginx/renuma-api-access.log;
    error_log /var/log/nginx/renuma-api-error.log;

    # Handle requests
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    # PHP-FPM Configuration
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
        fastcgi_hide_header X-Powered-By;
    }

    # Security
    location ~ /\.(?!well-known).* {
        deny all;
    }

    # File upload limit
    client_max_body_size 20M;
}
```

### Frontend Configuration

```nginx
# /etc/nginx/sites-available/renuma-frontend
server {
    listen 80;
    server_name renumarealestate.com www.renumarealestate.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name renumarealestate.com www.renumarealestate.com;

    # SSL Configuration
    ssl_certificate /etc/letsencrypt/live/renumarealestate.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/renumarealestate.com/privkey.pem;

    # Security headers
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;

    # Logging
    access_log /var/log/nginx/renuma-frontend-access.log;
    error_log /var/log/nginx/renuma-frontend-error.log;

    # Proxy to Next.js
    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }

    # Static files cache
    location /_next/static {
        proxy_cache STATIC;
        proxy_pass http://localhost:3000;
        add_header Cache-Control "public, max-age=31536000, immutable";
    }
}
```

### Enable Sites

```bash
# Create symbolic links
sudo ln -s /etc/nginx/sites-available/renuma-api /etc/nginx/sites-enabled/
sudo ln -s /etc/nginx/sites-available/renuma-frontend /etc/nginx/sites-enabled/

# Test configuration
sudo nginx -t

# Restart Nginx
sudo systemctl restart nginx
```

---

## 🔒 SSL Certificate (Let's Encrypt)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx -y

# Obtain SSL certificates
sudo certbot --nginx -d renumarealestate.com -d www.renumarealestate.com
sudo certbot --nginx -d api.renumarealestate.com

# Auto-renewal test
sudo certbot renew --dry-run
```

---

## 💾 Backup Strategy

### 1. Database Backup Script

```bash
# Create backup script
sudo nano /usr/local/bin/renuma-backup.sh

# Add content:
#!/bin/bash
BACKUP_DIR="/backups/renuma"
DATE=$(date +%Y%m%d_%H%M%S)
mkdir -p $BACKUP_DIR

# Database backup
mysqldump -u renuma_user -p'password' renuma_db | gzip > $BACKUP_DIR/db_$DATE.sql.gz

# Files backup
tar -czf $BACKUP_DIR/files_$DATE.tar.gz /var/www/renuma-real-estate/backend/storage/app/public

# Keep only last 30 days
find $BACKUP_DIR -type f -mtime +30 -delete

# Make executable
sudo chmod +x /usr/local/bin/renuma-backup.sh

# Schedule daily backup
crontab -e
# Add: 0 2 * * * /usr/local/bin/renuma-backup.sh
```

---

## 📊 Monitoring & Logs

### View Logs

```bash
# Laravel logs
tail -f /var/www/renuma-real-estate/backend/storage/logs/laravel.log

# Nginx logs
tail -f /var/log/nginx/renuma-api-error.log
tail -f /var/log/nginx/renuma-frontend-error.log

# PM2 logs
pm2 logs renuma-frontend
```

---

## 🔧 Performance Optimization

### Enable Redis

```bash
# Install Redis
sudo apt install redis-server -y

# Start Redis
sudo systemctl start redis
sudo systemctl enable redis

# Update Laravel .env
CACHE_DRIVER=redis
QUEUE_CONNECTION=redis
SESSION_DRIVER=redis
```

### Enable OpCache

```bash
# Edit PHP config
sudo nano /etc/php/8.1/fpm/php.ini

# Add/update:
opcache.enable=1
opcache.memory_consumption=256
opcache.interned_strings_buffer=16
opcache.max_accelerated_files=10000
opcache.revalidate_freq=2

# Restart PHP-FPM
sudo systemctl restart php8.1-fpm
```

---

## ✅ Post-Deployment Checklist

- [ ] SSL certificates installed and working
- [ ] Database migrations completed
- [ ] Storage directories writable
- [ ] Email sending configured and tested
- [ ] Cron jobs scheduled
- [ ] Queue workers running
- [ ] Backup script configured
- [ ] Firewall configured (allow 80, 443)
- [ ] Redis running
- [ ] Google Maps API configured
- [ ] WhatsApp integration tested
- [ ] Admin panel accessible
- [ ] Contact forms working
- [ ] Property search functional
- [ ] Image uploads working

---

## 🆘 Troubleshooting

### Common Issues

**500 Internal Server Error**
```bash
# Check permissions
sudo chown -R www-data:www-data /var/www/renuma-real-estate
sudo chmod -R 755 /var/www/renuma-real-estate
sudo chmod -R 775 storage bootstrap/cache

# Clear cache
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

**Database Connection Error**
```bash
# Test database connection
mysql -u renuma_user -p renuma_db

# Check .env file
cat .env | grep DB_
```

**Frontend Not Loading**
```bash
# Check PM2 status
pm2 status

# Restart frontend
pm2 restart renuma-frontend

# Check logs
pm2 logs renuma-frontend
```

---

## 📞 Support

For deployment support, contact:
- Email: renumarealestate@gmail.com
- Phone: +8801315333100

---

**Deployment completed successfully! 🎉**
