<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Lead;
use App\Models\Property;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;

class LeadController extends Controller
{
    /**
     * Submit a new lead/inquiry
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'nullable|string|max:20',
            'message' => 'required|string',
            'property_id' => 'nullable|exists:properties,id',
            'inquiry_type' => 'nullable|in:general,property,viewing,callback',
            'subject' => 'nullable|string|max:255',
        ]);

        try {
            // Create lead
            $lead = Lead::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'message' => $request->message,
                'property_id' => $request->property_id,
                'inquiry_type' => $request->inquiry_type ?? 'general',
                'subject' => $request->subject,
                'budget_min' => $request->budget_min,
                'budget_max' => $request->budget_max,
                'preferred_location' => $request->preferred_location,
                'property_type' => $request->property_type,
                'source' => $request->source ?? 'website',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'status' => 'new',
            ]);

            // Increment property inquiry count
            if ($request->property_id) {
                $property = Property::find($request->property_id);
                if ($property) {
                    $property->incrementInquiries();
                    
                    // Assign to property agent if available
                    if ($property->agent_id) {
                        $lead->update(['agent_id' => $property->agent_id]);
                    }
                }
            }

            // Send email notification
            $this->sendLeadNotification($lead);

            // Send auto-response to customer
            $this->sendAutoResponse($lead);

            return response()->json([
                'success' => true,
                'message' => 'Thank you for your inquiry. We will contact you shortly.',
                'data' => $lead
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error submitting inquiry. Please try again.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Submit viewing appointment request
     */
    public function requestViewing(Request $request)
    {
        $request->validate([
            'property_id' => 'required|exists:properties,id',
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'preferred_date' => 'required|date|after:today',
            'preferred_time' => 'required',
            'message' => 'nullable|string',
        ]);

        try {
            $property = Property::findOrFail($request->property_id);

            $appointment = \App\Models\ViewingAppointment::create([
                'property_id' => $request->property_id,
                'agent_id' => $property->agent_id,
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'preferred_date' => $request->preferred_date,
                'preferred_time' => $request->preferred_time,
                'message' => $request->message,
                'status' => 'pending',
            ]);

            // Create a lead as well
            $lead = Lead::create([
                'property_id' => $request->property_id,
                'agent_id' => $property->agent_id,
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'message' => "Viewing request for {$request->preferred_date} at {$request->preferred_time}. " . $request->message,
                'inquiry_type' => 'viewing',
                'status' => 'new',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            // Send notifications
            $this->sendViewingRequestNotification($appointment);

            return response()->json([
                'success' => true,
                'message' => 'Viewing appointment request submitted successfully',
                'data' => $appointment
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error submitting viewing request',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Submit callback request
     */
    public function requestCallback(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'property_id' => 'nullable|exists:properties,id',
            'preferred_time' => 'nullable|string',
        ]);

        try {
            $lead = Lead::create([
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'property_id' => $request->property_id,
                'message' => "Callback request. Preferred time: " . ($request->preferred_time ?? 'Any time'),
                'inquiry_type' => 'callback',
                'status' => 'new',
                'priority' => 'high',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);

            $this->sendCallbackNotification($lead);

            return response()->json([
                'success' => true,
                'message' => 'Callback request received. We will call you soon.',
                'data' => $lead
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error submitting callback request'
            ], 500);
        }
    }

    /**
     * Send lead notification to admin/agent
     */
    private function sendLeadNotification($lead)
    {
        try {
            $adminEmail = config('mail.admin_email', 'renumarealestate@gmail.com');
            
            Mail::send('emails.new-lead', ['lead' => $lead], function ($message) use ($adminEmail, $lead) {
                $message->to($adminEmail)
                       ->subject('New Inquiry - Renuma Real Estate');
            });

            // If assigned to agent, notify them too
            if ($lead->agent) {
                Mail::send('emails.new-lead', ['lead' => $lead], function ($message) use ($lead) {
                    $message->to($lead->agent->email)
                           ->subject('New Lead Assigned to You');
                });
            }
        } catch (\Exception $e) {
            \Log::error('Lead notification failed: ' . $e->getMessage());
        }
    }

    /**
     * Send auto-response to customer
     */
    private function sendAutoResponse($lead)
    {
        try {
            Mail::send('emails.lead-confirmation', ['lead' => $lead], function ($message) use ($lead) {
                $message->to($lead->email)
                       ->subject('Thank You for Your Inquiry - Renuma Real Estate');
            });
        } catch (\Exception $e) {
            \Log::error('Auto response failed: ' . $e->getMessage());
        }
    }

    /**
     * Send viewing request notification
     */
    private function sendViewingRequestNotification($appointment)
    {
        try {
            $adminEmail = config('mail.admin_email', 'renumarealestate@gmail.com');
            
            Mail::send('emails.viewing-request', ['appointment' => $appointment], function ($message) use ($adminEmail) {
                $message->to($adminEmail)
                       ->subject('New Viewing Request - Renuma Real Estate');
            });
        } catch (\Exception $e) {
            \Log::error('Viewing notification failed: ' . $e->getMessage());
        }
    }

    /**
     * Send callback notification
     */
    private function sendCallbackNotification($lead)
    {
        try {
            $adminEmail = config('mail.admin_email', 'renumarealestate@gmail.com');
            
            Mail::send('emails.callback-request', ['lead' => $lead], function ($message) use ($adminEmail) {
                $message->to($adminEmail)
                       ->subject('Urgent: Callback Request - Renuma Real Estate');
            });
        } catch (\Exception $e) {
            \Log::error('Callback notification failed: ' . $e->getMessage());
        }
    }
}
