<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\Lead;
use App\Models\Agent;
use App\Models\User;
use App\Models\BlogPost;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Get dashboard statistics and analytics
     */
    public function index()
    {
        try {
            $stats = [
                'overview' => $this->getOverviewStats(),
                'properties' => $this->getPropertyStats(),
                'leads' => $this->getLeadStats(),
                'revenue' => $this->getRevenueStats(),
                'recent_activities' => $this->getRecentActivities(),
                'top_agents' => $this->getTopAgents(),
                'popular_locations' => $this->getPopularLocations(),
                'monthly_stats' => $this->getMonthlyStats(),
            ];

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching dashboard data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get overview statistics
     */
    private function getOverviewStats()
    {
        return [
            'total_properties' => Property::count(),
            'active_properties' => Property::where('status', 'available')->count(),
            'sold_properties' => Property::where('status', 'sold')->count(),
            'rented_properties' => Property::where('status', 'rented')->count(),
            'total_leads' => Lead::count(),
            'new_leads' => Lead::where('status', 'new')->count(),
            'total_agents' => Agent::where('status', 'active')->count(),
            'total_users' => User::count(),
            'featured_properties' => Property::where('is_featured', true)->count(),
            'blog_posts' => BlogPost::where('status', 'published')->count(),
        ];
    }

    /**
     * Get property statistics
     */
    private function getPropertyStats()
    {
        $today = Carbon::today();
        $lastWeek = Carbon::today()->subDays(7);
        $lastMonth = Carbon::today()->subDays(30);

        return [
            'total' => Property::count(),
            'available' => Property::where('status', 'available')->count(),
            'for_sale' => Property::whereIn('purpose', ['sale', 'both'])->count(),
            'for_rent' => Property::whereIn('purpose', ['rent', 'both'])->count(),
            'featured' => Property::where('is_featured', true)->count(),
            'verified' => Property::where('is_verified', true)->count(),
            'added_today' => Property::whereDate('created_at', $today)->count(),
            'added_this_week' => Property::where('created_at', '>=', $lastWeek)->count(),
            'added_this_month' => Property::where('created_at', '>=', $lastMonth)->count(),
            'total_views' => Property::sum('views'),
            'total_inquiries' => Property::sum('inquiries'),
            'by_type' => Property::select('property_type_id', DB::raw('count(*) as count'))
                ->with('propertyType:id,name')
                ->groupBy('property_type_id')
                ->get(),
            'by_status' => Property::select('status', DB::raw('count(*) as count'))
                ->groupBy('status')
                ->get(),
        ];
    }

    /**
     * Get lead statistics
     */
    private function getLeadStats()
    {
        $today = Carbon::today();
        $lastWeek = Carbon::today()->subDays(7);

        return [
            'total' => Lead::count(),
            'new' => Lead::where('status', 'new')->count(),
            'contacted' => Lead::where('status', 'contacted')->count(),
            'qualified' => Lead::where('status', 'qualified')->count(),
            'converted' => Lead::where('status', 'converted')->count(),
            'lost' => Lead::where('status', 'lost')->count(),
            'today' => Lead::whereDate('created_at', $today)->count(),
            'this_week' => Lead::where('created_at', '>=', $lastWeek)->count(),
            'high_priority' => Lead::where('priority', 'high')->count(),
            'by_type' => Lead::select('inquiry_type', DB::raw('count(*) as count'))
                ->groupBy('inquiry_type')
                ->get(),
            'by_status' => Lead::select('status', DB::raw('count(*) as count'))
                ->groupBy('status')
                ->get(),
            'conversion_rate' => $this->getConversionRate(),
        ];
    }

    /**
     * Get revenue statistics
     */
    private function getRevenueStats()
    {
        $soldProperties = Property::where('status', 'sold')->get();
        $rentedProperties = Property::where('status', 'rented')->get();

        return [
            'total_sales_value' => $soldProperties->sum('price'),
            'total_rental_value' => $rentedProperties->sum('price'),
            'average_sale_price' => $soldProperties->avg('price'),
            'average_rental_price' => $rentedProperties->avg('price'),
            'properties_sold_count' => $soldProperties->count(),
            'properties_rented_count' => $rentedProperties->count(),
            'estimated_commission' => $soldProperties->sum('price') * 0.03, // 3% commission
        ];
    }

    /**
     * Get recent activities
     */
    private function getRecentActivities()
    {
        return [
            'recent_properties' => Property::with(['propertyType', 'location', 'agent'])
                ->latest()
                ->limit(5)
                ->get(),
            'recent_leads' => Lead::with('property')
                ->latest()
                ->limit(10)
                ->get(),
            'recent_sold' => Property::with(['propertyType', 'location'])
                ->where('status', 'sold')
                ->latest('updated_at')
                ->limit(5)
                ->get(),
        ];
    }

    /**
     * Get top performing agents
     */
    private function getTopAgents()
    {
        return Agent::withCount([
            'properties as total_properties',
            'properties as sold_properties' => function ($query) {
                $query->where('status', 'sold');
            },
            'properties as rented_properties' => function ($query) {
                $query->where('status', 'rented');
            },
            'leads as total_leads'
        ])
        ->withSum('properties', 'views')
        ->orderBy('sold_properties', 'desc')
        ->limit(10)
        ->get();
    }

    /**
     * Get popular locations
     */
    private function getPopularLocations()
    {
        return DB::table('locations')
            ->join('properties', 'locations.id', '=', 'properties.location_id')
            ->select('locations.name', 'locations.slug', DB::raw('count(properties.id) as property_count'))
            ->groupBy('locations.id', 'locations.name', 'locations.slug')
            ->orderBy('property_count', 'desc')
            ->limit(10)
            ->get();
    }

    /**
     * Get monthly statistics for charts
     */
    private function getMonthlyStats()
    {
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $months[] = [
                'month' => $date->format('M Y'),
                'properties_added' => Property::whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)
                    ->count(),
                'leads_received' => Lead::whereYear('created_at', $date->year)
                    ->whereMonth('created_at', $date->month)
                    ->count(),
                'properties_sold' => Property::where('status', 'sold')
                    ->whereYear('updated_at', $date->year)
                    ->whereMonth('updated_at', $date->month)
                    ->count(),
                'properties_rented' => Property::where('status', 'rented')
                    ->whereYear('updated_at', $date->year)
                    ->whereMonth('updated_at', $date->month)
                    ->count(),
            ];
        }

        return $months;
    }

    /**
     * Calculate lead conversion rate
     */
    private function getConversionRate()
    {
        $totalLeads = Lead::count();
        $convertedLeads = Lead::where('status', 'converted')->count();

        if ($totalLeads === 0) {
            return 0;
        }

        return round(($convertedLeads / $totalLeads) * 100, 2);
    }

    /**
     * Get analytics for specific date range
     */
    public function analytics(Request $request)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        try {
            $startDate = $request->start_date ?? Carbon::now()->subDays(30);
            $endDate = $request->end_date ?? Carbon::now();

            $analytics = [
                'properties' => [
                    'added' => Property::whereBetween('created_at', [$startDate, $endDate])->count(),
                    'sold' => Property::where('status', 'sold')
                        ->whereBetween('updated_at', [$startDate, $endDate])
                        ->count(),
                    'rented' => Property::where('status', 'rented')
                        ->whereBetween('updated_at', [$startDate, $endDate])
                        ->count(),
                ],
                'leads' => [
                    'total' => Lead::whereBetween('created_at', [$startDate, $endDate])->count(),
                    'converted' => Lead::where('status', 'converted')
                        ->whereBetween('created_at', [$startDate, $endDate])
                        ->count(),
                    'by_source' => Lead::whereBetween('created_at', [$startDate, $endDate])
                        ->select('source', DB::raw('count(*) as count'))
                        ->groupBy('source')
                        ->get(),
                ],
                'traffic' => [
                    'total_views' => Property::whereBetween('created_at', [$startDate, $endDate])
                        ->sum('views'),
                    'avg_views_per_property' => Property::whereBetween('created_at', [$startDate, $endDate])
                        ->avg('views'),
                ],
            ];

            return response()->json([
                'success' => true,
                'data' => $analytics,
                'period' => [
                    'start' => $startDate,
                    'end' => $endDate,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching analytics'
            ], 500);
        }
    }

    /**
     * Export dashboard data
     */
    public function export(Request $request)
    {
        try {
            $type = $request->get('type', 'overview');

            switch ($type) {
                case 'properties':
                    $data = Property::with(['propertyType', 'location', 'agent'])->get();
                    break;
                case 'leads':
                    $data = Lead::with('property')->get();
                    break;
                case 'agents':
                    $data = Agent::withCount('properties')->get();
                    break;
                default:
                    $data = $this->getOverviewStats();
            }

            return response()->json([
                'success' => true,
                'data' => $data,
                'export_date' => now()->toDateTimeString()
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Export failed'
            ], 500);
        }
    }
}
