<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\PropertyType;
use App\Models\Location;
use App\Models\Agent;
use App\Models\Feature;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PropertyController extends Controller
{
    /**
     * Display a listing of all properties
     */
    public function index(Request $request)
    {
        try {
            $query = Property::with(['propertyType', 'location', 'agent']);

            // Search
            if ($request->filled('search')) {
                $query->search($request->search);
            }

            // Status filter
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }

            // Purpose filter
            if ($request->filled('purpose')) {
                $query->where('purpose', $request->purpose);
            }

            // Type filter
            if ($request->filled('type')) {
                $query->where('property_type_id', $request->type);
            }

            // Agent filter
            if ($request->filled('agent')) {
                $query->where('agent_id', $request->agent);
            }

            // Date range
            if ($request->filled('date_from')) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }
            if ($request->filled('date_to')) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // Sort
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            $query->orderBy($sortBy, $sortOrder);

            $properties = $query->paginate(20);

            return response()->json([
                'success' => true,
                'data' => $properties,
                'stats' => [
                    'total' => Property::count(),
                    'available' => Property::where('status', 'available')->count(),
                    'sold' => Property::where('status', 'sold')->count(),
                    'rented' => Property::where('status', 'rented')->count(),
                    'featured' => Property::where('is_featured', true)->count(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching properties',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created property
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'property_type_id' => 'required|exists:property_types,id',
            'location_id' => 'required|exists:locations,id',
            'price' => 'required|numeric|min:0',
            'purpose' => 'required|in:sale,rent,both',
            'status' => 'required|in:available,sold,rented,pending',
            'bedrooms' => 'nullable|integer|min:0',
            'bathrooms' => 'nullable|integer|min:0',
            'area_sqft' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'address' => 'nullable|string|max:500',
            'featured_image' => 'nullable|image|max:5120',
            'images.*' => 'nullable|image|max:5120',
        ]);

        DB::beginTransaction();

        try {
            // Handle featured image upload
            $featuredImagePath = null;
            if ($request->hasFile('featured_image')) {
                $featuredImagePath = $request->file('featured_image')
                    ->store('properties', 'public');
            }

            // Create property
            $property = Property::create([
                'title' => $request->title,
                'slug' => Str::slug($request->title),
                'property_type_id' => $request->property_type_id,
                'location_id' => $request->location_id,
                'agent_id' => $request->agent_id,
                'description' => $request->description,
                'short_description' => $request->short_description,
                'address' => $request->address,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'zip_code' => $request->zip_code,
                'price' => $request->price,
                'price_type' => $request->price_type ?? 'fixed',
                'currency' => $request->currency ?? 'BDT',
                'status' => $request->status,
                'purpose' => $request->purpose,
                'bedrooms' => $request->bedrooms,
                'bathrooms' => $request->bathrooms,
                'area_sqft' => $request->area_sqft,
                'area_type' => $request->area_type,
                'floors' => $request->floors,
                'year_built' => $request->year_built,
                'parking_spaces' => $request->parking_spaces,
                'featured_image' => $featuredImagePath,
                'video_url' => $request->video_url,
                'virtual_tour_url' => $request->virtual_tour_url,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords,
                'is_featured' => $request->boolean('is_featured'),
                'is_verified' => $request->boolean('is_verified'),
                'is_urgent' => $request->boolean('is_urgent'),
                'published_at' => $request->published_at ?? now(),
            ]);

            // Handle multiple images
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $index => $image) {
                    $path = $image->store('properties/gallery', 'public');
                    $property->images()->create([
                        'image_path' => $path,
                        'sort_order' => $index,
                    ]);
                }
            }

            // Handle floor plan
            if ($request->hasFile('floor_plan')) {
                $floorPlanPath = $request->file('floor_plan')
                    ->store('properties/floor-plans', 'public');
                $property->update(['floor_plan' => $floorPlanPath]);
            }

            // Attach features
            if ($request->filled('features')) {
                $property->features()->sync($request->features);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Property created successfully',
                'data' => $property->load(['propertyType', 'location', 'agent', 'images', 'features'])
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating property',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified property
     */
    public function show($id)
    {
        try {
            $property = Property::with([
                'propertyType',
                'location',
                'agent',
                'images',
                'features',
                'leads' => function ($query) {
                    $query->latest()->limit(10);
                }
            ])->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $property
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Property not found'
            ], 404);
        }
    }

    /**
     * Update the specified property
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'property_type_id' => 'required|exists:property_types,id',
            'location_id' => 'required|exists:locations,id',
            'price' => 'required|numeric|min:0',
            'purpose' => 'required|in:sale,rent,both',
            'status' => 'required|in:available,sold,rented,pending',
        ]);

        DB::beginTransaction();

        try {
            $property = Property::findOrFail($id);

            // Handle featured image upload
            if ($request->hasFile('featured_image')) {
                // Delete old image
                if ($property->featured_image) {
                    Storage::disk('public')->delete($property->featured_image);
                }
                $property->featured_image = $request->file('featured_image')
                    ->store('properties', 'public');
            }

            // Update property
            $property->update([
                'title' => $request->title,
                'property_type_id' => $request->property_type_id,
                'location_id' => $request->location_id,
                'agent_id' => $request->agent_id,
                'description' => $request->description,
                'short_description' => $request->short_description,
                'address' => $request->address,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'zip_code' => $request->zip_code,
                'price' => $request->price,
                'price_type' => $request->price_type ?? 'fixed',
                'status' => $request->status,
                'purpose' => $request->purpose,
                'bedrooms' => $request->bedrooms,
                'bathrooms' => $request->bathrooms,
                'area_sqft' => $request->area_sqft,
                'area_type' => $request->area_type,
                'floors' => $request->floors,
                'year_built' => $request->year_built,
                'parking_spaces' => $request->parking_spaces,
                'video_url' => $request->video_url,
                'virtual_tour_url' => $request->virtual_tour_url,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'meta_keywords' => $request->meta_keywords,
                'is_featured' => $request->boolean('is_featured'),
                'is_verified' => $request->boolean('is_verified'),
                'is_urgent' => $request->boolean('is_urgent'),
            ]);

            // Handle new images
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $index => $image) {
                    $path = $image->store('properties/gallery', 'public');
                    $property->images()->create([
                        'image_path' => $path,
                        'sort_order' => $property->images()->count() + $index,
                    ]);
                }
            }

            // Update features
            if ($request->has('features')) {
                $property->features()->sync($request->features);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Property updated successfully',
                'data' => $property->load(['propertyType', 'location', 'agent', 'images', 'features'])
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating property',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified property
     */
    public function destroy($id)
    {
        try {
            $property = Property::findOrFail($id);

            // Delete images
            if ($property->featured_image) {
                Storage::disk('public')->delete($property->featured_image);
            }

            foreach ($property->images as $image) {
                Storage::disk('public')->delete($image->image_path);
            }

            if ($property->floor_plan) {
                Storage::disk('public')->delete($property->floor_plan);
            }

            $property->delete();

            return response()->json([
                'success' => true,
                'message' => 'Property deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting property',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured($id)
    {
        try {
            $property = Property::findOrFail($id);
            $property->is_featured = !$property->is_featured;
            $property->save();

            return response()->json([
                'success' => true,
                'message' => 'Featured status updated',
                'is_featured' => $property->is_featured
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating featured status'
            ], 500);
        }
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,feature,unfeature,publish,unpublish',
            'ids' => 'required|array',
            'ids.*' => 'exists:properties,id'
        ]);

        try {
            $properties = Property::whereIn('id', $request->ids);

            switch ($request->action) {
                case 'delete':
                    $properties->delete();
                    $message = 'Properties deleted successfully';
                    break;
                case 'feature':
                    $properties->update(['is_featured' => true]);
                    $message = 'Properties marked as featured';
                    break;
                case 'unfeature':
                    $properties->update(['is_featured' => false]);
                    $message = 'Properties unmarked as featured';
                    break;
                case 'publish':
                    $properties->update(['published_at' => now()]);
                    $message = 'Properties published';
                    break;
                case 'unpublish':
                    $properties->update(['published_at' => null]);
                    $message = 'Properties unpublished';
                    break;
            }

            return response()->json([
                'success' => true,
                'message' => $message
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Bulk action failed'
            ], 500);
        }
    }

    /**
     * Get property options for forms
     */
    public function formOptions()
    {
        return response()->json([
            'success' => true,
            'data' => [
                'types' => PropertyType::active()->get(),
                'locations' => Location::active()->get(),
                'agents' => Agent::active()->get(),
                'features' => Feature::all()->groupBy('category'),
            ]
        ]);
    }
}
