<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Property extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'title',
        'slug',
        'property_type_id',
        'location_id',
        'agent_id',
        'description',
        'short_description',
        'address',
        'latitude',
        'longitude',
        'zip_code',
        'price',
        'price_type',
        'currency',
        'status',
        'purpose',
        'bedrooms',
        'bathrooms',
        'area_sqft',
        'area_type',
        'floors',
        'year_built',
        'parking_spaces',
        'featured_image',
        'video_url',
        'virtual_tour_url',
        'floor_plan',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'is_featured',
        'is_verified',
        'is_urgent',
        'views',
        'favorites',
        'inquiries',
        'published_at',
        'expires_at'
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'area_sqft' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'is_featured' => 'boolean',
        'is_verified' => 'boolean',
        'is_urgent' => 'boolean',
        'published_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    protected $appends = ['formatted_price', 'location_full'];

    // Automatically generate slug
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($property) {
            if (empty($property->slug)) {
                $property->slug = Str::slug($property->title);
            }
        });

        static::updating(function ($property) {
            if ($property->isDirty('title')) {
                $property->slug = Str::slug($property->title);
            }
        });
    }

    // Relationships
    public function propertyType()
    {
        return $this->belongsTo(PropertyType::class);
    }

    public function location()
    {
        return $this->belongsTo(Location::class);
    }

    public function agent()
    {
        return $this->belongsTo(Agent::class);
    }

    public function images()
    {
        return $this->hasMany(PropertyImage::class)->orderBy('sort_order');
    }

    public function features()
    {
        return $this->belongsToMany(Feature::class, 'property_features');
    }

    public function leads()
    {
        return $this->hasMany(Lead::class);
    }

    public function favorites()
    {
        return $this->hasMany(Favorite::class);
    }

    public function viewingAppointments()
    {
        return $this->hasMany(ViewingAppointment::class);
    }

    // Scopes
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeAvailable($query)
    {
        return $query->where('status', 'available');
    }

    public function scopeForSale($query)
    {
        return $query->whereIn('purpose', ['sale', 'both']);
    }

    public function scopeForRent($query)
    {
        return $query->whereIn('purpose', ['rent', 'both']);
    }

    public function scopePublished($query)
    {
        return $query->whereNotNull('published_at')
                    ->where('published_at', '<=', now());
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%")
              ->orWhere('address', 'LIKE', "%{$search}%");
        });
    }

    public function scopeFilterByType($query, $typeId)
    {
        if ($typeId) {
            return $query->where('property_type_id', $typeId);
        }
        return $query;
    }

    public function scopeFilterByLocation($query, $locationId)
    {
        if ($locationId) {
            return $query->where('location_id', $locationId);
        }
        return $query;
    }

    public function scopeFilterByPrice($query, $minPrice, $maxPrice)
    {
        if ($minPrice) {
            $query->where('price', '>=', $minPrice);
        }
        if ($maxPrice) {
            $query->where('price', '<=', $maxPrice);
        }
        return $query;
    }

    public function scopeFilterByBedrooms($query, $bedrooms)
    {
        if ($bedrooms) {
            return $query->where('bedrooms', '>=', $bedrooms);
        }
        return $query;
    }

    public function scopeFilterByBathrooms($query, $bathrooms)
    {
        if ($bathrooms) {
            return $query->where('bathrooms', '>=', $bathrooms);
        }
        return $query;
    }

    // Accessors
    public function getFormattedPriceAttribute()
    {
        return $this->currency . ' ' . number_format($this->price, 2);
    }

    public function getLocationFullAttribute()
    {
        if ($this->location) {
            return $this->location->name . ', ' . ($this->location->parent ? $this->location->parent->name : '');
        }
        return $this->address;
    }

    // Helper methods
    public function incrementViews()
    {
        $this->increment('views');
    }

    public function incrementInquiries()
    {
        $this->increment('inquiries');
    }

    public function incrementFavorites()
    {
        $this->increment('favorites');
    }

    public function decrementFavorites()
    {
        $this->decrement('favorites');
    }

    public function isAvailable()
    {
        return $this->status === 'available';
    }

    public function isFeatured()
    {
        return $this->is_featured;
    }

    public function getStatusBadge()
    {
        $badges = [
            'available' => '<span class="badge bg-success">Available</span>',
            'sold' => '<span class="badge bg-danger">Sold</span>',
            'rented' => '<span class="badge bg-warning">Rented</span>',
            'pending' => '<span class="badge bg-info">Pending</span>',
        ];

        return $badges[$this->status] ?? '';
    }

    public function getPurposeBadge()
    {
        $badges = [
            'sale' => '<span class="badge bg-primary">For Sale</span>',
            'rent' => '<span class="badge bg-secondary">For Rent</span>',
            'both' => '<span class="badge bg-info">Sale/Rent</span>',
        ];

        return $badges[$this->purpose] ?? '';
    }
}
