import React, { useState } from 'react';
import { Heart, MapPin, Bed, Bath, Maximize, Eye, Phone } from 'lucide-react';
import { motion } from 'framer-motion';

export default function PropertyCard({ property }) {
  const [isFavorite, setIsFavorite] = useState(false);

  const toggleFavorite = async () => {
    try {
      await fetch('/api/v1/favorites/toggle', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ property_id: property.id })
      });
      setIsFavorite(!isFavorite);
    } catch (error) {
      console.error('Error toggling favorite:', error);
    }
  };

  const formatPrice = (price, currency = 'BDT') => {
    if (currency === 'BDT') {
      return `৳ ${(price / 100000).toFixed(2)} Lac`;
    }
    return `${currency} ${price.toLocaleString()}`;
  };

  const getStatusBadge = (status) => {
    const badges = {
      available: 'bg-green-500',
      sold: 'bg-red-500',
      rented: 'bg-yellow-500',
      pending: 'bg-blue-500'
    };
    return badges[status] || 'bg-gray-500';
  };

  return (
    <motion.div
      className="property-card bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 group"
      whileHover={{ y: -5 }}
    >
      {/* Image Container */}
      <div className="relative h-72 overflow-hidden">
        <img
          src={property.featured_image || property.images?.[0]?.image_path || '/images/placeholder.jpg'}
          alt={property.title}
          className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
        />

        {/* Overlay */}
        <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>

        {/* Status Badge */}
        <div className="absolute top-4 left-4">
          <span className={`px-3 py-1 text-xs font-semibold text-white rounded-full ${getStatusBadge(property.status)}`}>
            {property.status.charAt(0).toUpperCase() + property.status.slice(1)}
          </span>
        </div>

        {/* Purpose Badge */}
        <div className="absolute top-4 right-4">
          <span className="px-3 py-1 text-xs font-semibold bg-gold-500 text-white rounded-full">
            For {property.purpose.charAt(0).toUpperCase() + property.purpose.slice(1)}
          </span>
        </div>

        {/* Favorite Button */}
        <button
          onClick={toggleFavorite}
          className="absolute bottom-4 right-4 w-10 h-10 bg-white rounded-full flex items-center justify-center shadow-lg hover:bg-gold-500 hover:text-white transition-all duration-300 opacity-0 group-hover:opacity-100"
        >
          <Heart 
            className={`w-5 h-5 ${isFavorite ? 'fill-current text-red-500' : ''}`}
          />
        </button>

        {/* Quick View */}
        <div className="absolute bottom-4 left-4 flex gap-2 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
          <div className="flex items-center gap-1 bg-white/90 backdrop-blur-sm px-3 py-1 rounded-full text-sm">
            <Eye className="w-4 h-4 text-gray-600" />
            <span className="text-gray-700">{property.views || 0}</span>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="p-6">
        {/* Price */}
        <div className="mb-3">
          <h3 className="text-2xl font-bold text-gray-900">
            {formatPrice(property.price, property.currency)}
          </h3>
          <p className="text-sm text-gray-500">
            {property.price_type === 'negotiable' ? 'Negotiable' : ''}
          </p>
        </div>

        {/* Title */}
        <h4 className="text-xl font-semibold text-gray-900 mb-2 line-clamp-2 hover:text-gold-500 transition-colors">
          <a href={`/properties/${property.slug}`}>
            {property.title}
          </a>
        </h4>

        {/* Location */}
        <div className="flex items-center gap-2 text-gray-600 mb-4">
          <MapPin className="w-4 h-4 flex-shrink-0" />
          <span className="text-sm line-clamp-1">
            {property.location?.name || property.address}
          </span>
        </div>

        {/* Property Details */}
        <div className="flex items-center gap-4 mb-4 pb-4 border-b border-gray-200">
          {property.bedrooms && (
            <div className="flex items-center gap-2 text-gray-600">
              <Bed className="w-4 h-4" />
              <span className="text-sm font-medium">{property.bedrooms}</span>
            </div>
          )}
          {property.bathrooms && (
            <div className="flex items-center gap-2 text-gray-600">
              <Bath className="w-4 h-4" />
              <span className="text-sm font-medium">{property.bathrooms}</span>
            </div>
          )}
          {property.area_sqft && (
            <div className="flex items-center gap-2 text-gray-600">
              <Maximize className="w-4 h-4" />
              <span className="text-sm font-medium">{property.area_sqft} sqft</span>
            </div>
          )}
        </div>

        {/* Agent Info */}
        {property.agent && (
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <img
                src={property.agent.avatar || '/images/default-avatar.jpg'}
                alt={property.agent.name}
                className="w-10 h-10 rounded-full object-cover"
              />
              <div>
                <p className="text-sm font-medium text-gray-900">{property.agent.name}</p>
                <p className="text-xs text-gray-500">{property.agent.designation || 'Agent'}</p>
              </div>
            </div>

            {/* Contact Button */}
            {property.agent.phone && (
              <a
                href={`tel:${property.agent.phone}`}
                className="w-10 h-10 bg-gold-500 rounded-full flex items-center justify-center text-white hover:bg-gold-600 transition-colors"
              >
                <Phone className="w-5 h-5" />
              </a>
            )}
          </div>
        )}

        {/* Action Buttons */}
        <div className="mt-4 grid grid-cols-2 gap-3">
          <a
            href={`/properties/${property.slug}`}
            className="text-center py-2 px-4 border-2 border-gray-900 text-gray-900 rounded-lg hover:bg-gray-900 hover:text-white transition-all duration-300 font-semibold text-sm"
          >
            View Details
          </a>
          <a
            href={`https://wa.me/8801315333100?text=I'm interested in ${property.title}`}
            target="_blank"
            rel="noopener noreferrer"
            className="text-center py-2 px-4 bg-gold-500 text-white rounded-lg hover:bg-gold-600 transition-all duration-300 font-semibold text-sm"
          >
            WhatsApp
          </a>
        </div>
      </div>

      {/* Featured Badge */}
      {property.is_featured && (
        <div className="absolute top-0 left-0">
          <div className="bg-gold-500 text-white text-xs font-bold px-3 py-1 transform -rotate-45 -translate-x-8 translate-y-4">
            Featured
          </div>
        </div>
      )}

      {/* Urgent Badge */}
      {property.is_urgent && (
        <div className="absolute top-0 right-0">
          <div className="bg-red-500 text-white text-xs font-bold px-3 py-1 transform rotate-45 translate-x-8 translate-y-4">
            Urgent
          </div>
        </div>
      )}
    </motion.div>
  );
}
