import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Search, MapPin, Home, DollarSign, Phone, Mail, MessageCircle } from 'lucide-react';
import PropertyCard from '../components/properties/PropertyCard';
import AgentCard from '../components/home/AgentCard';
import TestimonialSlider from '../components/home/TestimonialSlider';
import StatsCounter from '../components/home/StatsCounter';
import FeaturedLocations from '../components/home/FeaturedLocations';

export default function HomePage() {
  const [featuredProperties, setFeaturedProperties] = useState([]);
  const [latestProperties, setLatestProperties] = useState([]);
  const [agents, setAgents] = useState([]);
  const [testimonials, setTestimonials] = useState([]);
  const [stats, setStats] = useState({});
  const [searchQuery, setSearchQuery] = useState({
    location: '',
    propertyType: '',
    purpose: 'sale',
    minPrice: '',
    maxPrice: ''
  });

  useEffect(() => {
    fetchHomeData();
  }, []);

  const fetchHomeData = async () => {
    try {
      // Fetch featured properties
      const featuredRes = await fetch('/api/v1/properties/featured');
      const featuredData = await featuredRes.json();
      if (featuredData.success) {
        setFeaturedProperties(featuredData.data);
      }

      // Fetch latest properties
      const latestRes = await fetch('/api/v1/properties/latest');
      const latestData = await latestRes.json();
      if (latestData.success) {
        setLatestProperties(latestData.data);
      }

      // Fetch agents
      const agentsRes = await fetch('/api/v1/agents/featured');
      const agentsData = await agentsRes.json();
      if (agentsData.success) {
        setAgents(agentsData.data);
      }

      // Fetch testimonials
      const testimonialsRes = await fetch('/api/v1/testimonials');
      const testimonialsData = await testimonialsRes.json();
      if (testimonialsData.success) {
        setTestimonials(testimonialsData.data);
      }

      // Fetch statistics
      const statsRes = await fetch('/api/v1/properties/statistics');
      const statsData = await statsRes.json();
      if (statsData.success) {
        setStats(statsData.data);
      }
    } catch (error) {
      console.error('Error fetching home data:', error);
    }
  };

  const handleSearch = (e) => {
    e.preventDefault();
    // Navigate to properties page with search params
    const params = new URLSearchParams(searchQuery).toString();
    window.location.href = `/properties?${params}`;
  };

  return (
    <div className="homepage">
      {/* Hero Section */}
      <section className="hero-section relative h-screen flex items-center justify-center overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="/images/hero-bg.jpg" 
            alt="Luxury Real Estate"
            className="w-full h-full object-cover"
          />
          <div className="absolute inset-0 bg-gradient-to-r from-gray-900/90 via-gray-900/70 to-transparent"></div>
        </div>

        <div className="container mx-auto px-4 z-10 relative">
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
            className="max-w-4xl"
          >
            <h1 className="text-5xl md:text-7xl font-bold text-white mb-6 leading-tight">
              Let's Find Your <br />
              <span className="text-gold-400">Dream House</span>
            </h1>
            <p className="text-xl text-gray-200 mb-12 max-w-2xl">
              Discover exceptional properties with Renuma Real Estate - Your trusted partner in finding the perfect home.
            </p>

            {/* Search Box */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.8, delay: 0.2 }}
              className="bg-white rounded-2xl shadow-2xl p-6 md:p-8"
            >
              <form onSubmit={handleSearch}>
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
                  {/* Location */}
                  <div className="relative">
                    <MapPin className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
                    <input
                      type="text"
                      placeholder="Location"
                      className="w-full pl-11 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-gold-400 focus:border-transparent"
                      value={searchQuery.location}
                      onChange={(e) => setSearchQuery({...searchQuery, location: e.target.value})}
                    />
                  </div>

                  {/* Property Type */}
                  <div className="relative">
                    <Home className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
                    <select
                      className="w-full pl-11 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-gold-400 focus:border-transparent appearance-none"
                      value={searchQuery.propertyType}
                      onChange={(e) => setSearchQuery({...searchQuery, propertyType: e.target.value})}
                    >
                      <option value="">Property Type</option>
                      <option value="apartment">Apartment</option>
                      <option value="villa">Villa</option>
                      <option value="land">Land</option>
                      <option value="commercial">Commercial</option>
                    </select>
                  </div>

                  {/* Price Range */}
                  <div className="relative">
                    <DollarSign className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
                    <input
                      type="text"
                      placeholder="Price Range"
                      className="w-full pl-11 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-gold-400 focus:border-transparent"
                      value={searchQuery.minPrice}
                      onChange={(e) => setSearchQuery({...searchQuery, minPrice: e.target.value})}
                    />
                  </div>

                  {/* Search Button */}
                  <button
                    type="submit"
                    className="bg-gold-500 hover:bg-gold-600 text-white font-semibold py-3 px-6 rounded-lg transition-all duration-300 flex items-center justify-center gap-2"
                  >
                    <Search className="w-5 h-5" />
                    Search
                  </button>
                </div>

                {/* Purpose Toggle */}
                <div className="flex gap-4">
                  <button
                    type="button"
                    className={`flex-1 py-2 px-4 rounded-lg font-medium transition-all ${
                      searchQuery.purpose === 'sale' 
                        ? 'bg-gray-900 text-white' 
                        : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                    }`}
                    onClick={() => setSearchQuery({...searchQuery, purpose: 'sale'})}
                  >
                    For Sale
                  </button>
                  <button
                    type="button"
                    className={`flex-1 py-2 px-4 rounded-lg font-medium transition-all ${
                      searchQuery.purpose === 'rent' 
                        ? 'bg-gray-900 text-white' 
                        : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                    }`}
                    onClick={() => setSearchQuery({...searchQuery, purpose: 'rent'})}
                  >
                    For Rent
                  </button>
                </div>
              </form>
            </motion.div>
          </motion.div>
        </div>

        {/* Scroll Indicator */}
        <div className="absolute bottom-10 left-1/2 -translate-x-1/2 z-10">
          <motion.div
            animate={{ y: [0, 10, 0] }}
            transition={{ repeat: Infinity, duration: 1.5 }}
            className="text-white opacity-70"
          >
            <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 14l-7 7m0 0l-7-7m7 7V3" />
            </svg>
          </motion.div>
        </div>
      </section>

      {/* Stats Section */}
      <section className="py-20 bg-gray-50">
        <div className="container mx-auto px-4">
          <StatsCounter stats={stats} />
        </div>
      </section>

      {/* Featured Properties */}
      <section className="py-20">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <motion.p
              initial={{ opacity: 0 }}
              whileInView={{ opacity: 1 }}
              className="text-gold-500 font-semibold mb-2"
            >
              Our Listings
            </motion.p>
            <motion.h2
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              className="text-4xl md:text-5xl font-bold text-gray-900"
            >
              Featured Properties
            </motion.h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {featuredProperties.map((property, index) => (
              <motion.div
                key={property.id}
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
              >
                <PropertyCard property={property} />
              </motion.div>
            ))}
          </div>

          <div className="text-center mt-12">
            <a
              href="/properties"
              className="inline-block bg-gray-900 text-white px-8 py-3 rounded-lg hover:bg-gold-500 transition-all duration-300 font-semibold"
            >
              View All Properties
            </a>
          </div>
        </div>
      </section>

      {/* About Section */}
      <section className="py-20 bg-gray-900 text-white">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-12 items-center">
            <motion.div
              initial={{ opacity: 0, x: -30 }}
              whileInView={{ opacity: 1, x: 0 }}
            >
              <img 
                src="/images/about.jpg" 
                alt="About Renuma Real Estate"
                className="rounded-2xl shadow-2xl"
              />
            </motion.div>

            <motion.div
              initial={{ opacity: 0, x: 30 }}
              whileInView={{ opacity: 1, x: 0 }}
            >
              <p className="text-gold-400 font-semibold mb-2">About Company</p>
              <h2 className="text-4xl font-bold mb-6">
                Welcome To <br />Renuma Real Estate
              </h2>
              <p className="text-gray-300 mb-6 leading-relaxed">
                It's our immense pleasure working to fulfill one of the basic human needs: home. 
                With our great team filled with brilliant minds, we try to fulfill dreams of many people. 
                That is why we consider this company "Home of Dreams".
              </p>

              <ul className="space-y-4 mb-8">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-gold-500 rounded-full flex items-center justify-center flex-shrink-0 mt-1">
                    <svg className="w-4 h-4 text-white" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <span className="text-gray-300">Proactively pontificate client-centered solutions</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-gold-500 rounded-full flex items-center justify-center flex-shrink-0 mt-1">
                    <svg className="w-4 h-4 text-white" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <span className="text-gray-300">Immediate 24/7 Emergency Support</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 bg-gold-500 rounded-full flex items-center justify-center flex-shrink-0 mt-1">
                    <svg className="w-4 h-4 text-white" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <span className="text-gray-300">Verified and Trusted Properties</span>
                </li>
              </ul>

              <a
                href="/about"
                className="inline-block bg-gold-500 text-white px-8 py-3 rounded-lg hover:bg-gold-600 transition-all duration-300 font-semibold"
              >
                Learn More
              </a>
            </motion.div>
          </div>
        </div>
      </section>

      {/* Featured Locations */}
      <section className="py-20">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <p className="text-gold-500 font-semibold mb-2">Our Property List</p>
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900">
              Top Locations For You
            </h2>
          </div>
          <FeaturedLocations />
        </div>
      </section>

      {/* Our Agents */}
      <section className="py-20 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <p className="text-gold-500 font-semibold mb-2">Our Expert</p>
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900">
              Meet Our Team
            </h2>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {agents.map((agent, index) => (
              <motion.div
                key={agent.id}
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
              >
                <AgentCard agent={agent} />
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* Testimonials */}
      <section className="py-20">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <p className="text-gold-500 font-semibold mb-2">Testimonials</p>
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900">
              What Our Clients Say?
            </h2>
          </div>
          <TestimonialSlider testimonials={testimonials} />
        </div>
      </section>

      {/* Contact CTA */}
      <section className="py-20 bg-gradient-to-r from-gray-900 to-gray-800 text-white">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto text-center">
            <h2 className="text-4xl md:text-5xl font-bold mb-6">
              Ready to Find Your Dream Property?
            </h2>
            <p className="text-xl text-gray-300 mb-8">
              Contact us today and let our expert team help you find the perfect home.
            </p>
            
            <div className="flex flex-wrap justify-center gap-6">
              <a
                href="tel:+8801315333100"
                className="flex items-center gap-3 bg-gold-500 hover:bg-gold-600 px-8 py-4 rounded-lg transition-all duration-300 font-semibold"
              >
                <Phone className="w-5 h-5" />
                Call Now
              </a>
              <a
                href="https://wa.me/8801315333100"
                target="_blank"
                rel="noopener noreferrer"
                className="flex items-center gap-3 bg-green-500 hover:bg-green-600 px-8 py-4 rounded-lg transition-all duration-300 font-semibold"
              >
                <MessageCircle className="w-5 h-5" />
                WhatsApp
              </a>
              <a
                href="/contact"
                className="flex items-center gap-3 bg-white hover:bg-gray-100 text-gray-900 px-8 py-4 rounded-lg transition-all duration-300 font-semibold"
              >
                <Mail className="w-5 h-5" />
                Contact Us
              </a>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}
